// --- הגדרות ---
const STORAGE_KEY = 'gem_db';
let debounceTimer = null;
let isWaitingForButton = false;

// =============================================================================
// חלק א': המגרד (Scraper) - שמירת ההנחיות מדף העריכה
// =============================================================================

function scrapeEditPage() {
    // בדיקה שאנחנו בדף עריכה של Gem
    const match = location.href.match(/\/gems\/edit\/([a-zA-Z0-9]+)/);
    if (!match) {
        return;
    }

    const gemId = match[1];

    // המתנה לכפתור העדכון להופיע
    isWaitingForButton = true;
    let attempts = 0;
    const interval = setInterval(() => {
        attempts++;

        if (attempts > 20) {
            clearInterval(interval);
            isWaitingForButton = false;
            return;
        }

        const spans = document.querySelectorAll('span.mdc-button__label');
        for (const span of spans) {
            const text = span.textContent.trim().toLowerCase();
            if (text.includes('עד')) {
                const updateBtn = span.closest('button') || span.closest('[role="button"]') || span.parentElement;
                if (updateBtn && !updateBtn.hasAttribute('data-gem-listener')) {
                    updateBtn.setAttribute('data-gem-listener', 'true');
                    updateBtn.addEventListener('click', () => {
                        // חיפוש תיבת ההנחיות
                        const selectors = [
                            'div[aria-label="כאן כותבים את ההנחיה"]', // עברית
                            'div[aria-label="Instructions"]',        // אנגלית
                            'div[aria-label="Instructions"]'         // גיבוי גנרי
                        ];

                        let instructionsBox = null;
                        for (const sel of selectors) {
                            instructionsBox = document.querySelector(sel);
                            if (instructionsBox) break;
                        }

                        if (instructionsBox) {
                            const text = instructionsBox.innerText.trim();
                            if (text.length > 0) {
                                // בדיקה שהתוסף עדיין פעיל
                                try {
                                    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
                                        chrome.storage.local.get(STORAGE_KEY, (result) => {
                                            try {
                                                if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
                                                const db = result[STORAGE_KEY] || {};
                                                db[gemId] = text;
                                                chrome.storage.local.set({ [STORAGE_KEY]: db }, () => {
                                                    showToast("הוראות נשמרו בהצלחה.");
                                                });
                                                }
                                            } catch (e) {
                                            }
                                        });
                                    }
                                } catch (e) {
                                }
                            } else {
                            }
                        } else {
                        }
                    });

                    clearInterval(interval);
                    isWaitingForButton = false;
                    return;
                }
            }
        }
    }, 500);
}

// =============================================================================
// חלק ב': המזריק (Injector) - הזרקה לשיחה חדשה
// =============================================================================

let hasInjected = false;
let hasClickedSend = false;
let lastChatUrl = "";

function handleChatPage() {
    // אם אנחנו בדף עריכה - יציאה
    if (location.href.includes('/gems/edit/')) {
        return;
    }

    // איפוס סטייט במעבר בין שיחות
    if (location.href !== lastChatUrl) {
        lastChatUrl = location.href;
        hasInjected = false;
        hasClickedSend = false;
    }

    if (hasInjected) {
        return;
    }

    // לוגיקה לזיהוי שיחה חדשה (למנוע הזרקה לשיחות היסטוריות)
    const chatHistoryParams = document.querySelectorAll('[data-message-author-role="user"]'); // נסה selector יותר מדויק

    if (chatHistoryParams.length > 0) {
        return;
    }

    // בדיקה נוספת: אם תיבת הקלט כבר מכילה טקסט, דלג (מניעת הזרקה כפולה)
    const inputBoxCheck = document.querySelector('.ql-editor.ql-blank') || 
                         document.querySelector('div[contenteditable="true"]');
    if (inputBoxCheck && inputBoxCheck.innerText.trim().length > 0) {
        return;
    }

    // חילוץ ה-ID
    const path = location.pathname;
    const parts = path.split('/').filter(p => p.length > 0);

    // מצפה למבנה: ["gem", "gemId"] בלבד (לא בשיחות קיימות)
    if (parts.length !== 2 || parts[0] !== 'gem') {
        return;
    }

    const gemId = parts[1]; // ההנחה שזה המזהה

    // בדיקה שהתוסף עדיין פעיל לפני שימוש ב-chrome APIs
    try {
        if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
            chrome.storage.local.get(STORAGE_KEY, (result) => {
                try {
                    // בדיקה נוספת שהתוסף עדיין פעיל בתוך ה-callback
                    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
                        const db = result[STORAGE_KEY] || {};
                        const savedInstructions = db[gemId];

                        if (savedInstructions) {
                            waitForInputAndInject(savedInstructions);
                        }
                    }
                } catch (e) {
                    // Extension context invalidated בתוך callback
                }
            });
        }
    } catch (e) {
        // Extension context invalidated - התוסף נטען מחדש
    }
}

function waitForInputAndInject(text) {
    let attempts = 0;
    // בדיקה כל 300ms עד 5 שניות
    const interval = setInterval(() => {
        attempts++;

        if (attempts > 15) {
            clearInterval(interval);
            return;
        }

        // חיפוש תיבת הטקסט - תמיכה ב-Selector העדכני של ג'מני
        const inputBox = document.querySelector('.ql-editor.ql-blank') || 
                         document.querySelector('div[contenteditable="true"]');

        if (inputBox) {
            // וידוא שהאלמנט נראה (לא מוסתר)
            if (inputBox.offsetParent === null) {
                return;
            }

            // הגנה נוספת: אם המשתמש כבר הקליד משהו, לא דורסים
            if (inputBox.innerText.trim().length > 0) {
                clearInterval(interval);
                return;
            }

            clearInterval(interval);
            injectText(inputBox, text);
        }
    }, 300);
}

function injectText(inputBox, text) {
    try {
        // 1. פוקוס
        inputBox.focus();

        // 2. הגדרת הטקסט ישירות (במקום הדבקה שיכולה להיכשל)
        inputBox.innerHTML = text;

        // 3. עירור קריטי! שליחת אירוע input כדי ש-Angular/React יזהו שינוי
        inputBox.dispatchEvent(new Event('input', { bubbles: true }));

        hasInjected = true;

        // השהייה קצרה כדי לתת ל-DOM זמן להתעדכן עם הטקסט שהודבק
        setTimeout(() => {
            attemptClickSend();
        }, 150);

    } catch (e) {
        console.error("Gemini Context Fixer: Injection failed", e);
    }
}

function attemptClickSend() {
    let clickAttempts = 0;
    const clickInterval = setInterval(() => {
        clickAttempts++;

        if (clickAttempts > 15) {
            clearInterval(clickInterval);
            return;
        }

        const sendBtn = document.querySelector('button[aria-label*="Send"]') ||
                        document.querySelector('button[aria-label*="שלח"]') ||
                        document.querySelector('button.send-button');

        // מציאת תיבת הקלט שוב כדי לבדוק את התוכן
        const inputBox = document.querySelector('.ql-editor') ||
                         document.querySelector('div[contenteditable="true"]');

        if (sendBtn && inputBox) {
            const label = (sendBtn.getAttribute('aria-label') || "").toLowerCase();
            const hasText = inputBox.innerText.trim().length > 0;
            const isEnabled = !sendBtn.disabled && !sendBtn.classList.contains('disabled');
            
            // הגנה: האם השיחה כבר רצה? (אם כן, אל תלחץ)
            if (label.includes("stop") || label.includes("עצור")) {
                clearInterval(clickInterval);
                return;
            }

            // הבדיקה הקריטית: האם הכפתור זמין *וגם* האם יש טקסט בפנים?
            if (!sendBtn.disabled && !sendBtn.classList.contains('disabled') && inputBox.innerText.trim().length > 0 && !hasClickedSend) {
                sendBtn.click();
                hasClickedSend = true;
                clearInterval(clickInterval);
                showToast("הוראות נשלחו בהצלחה.");
            } else {
                if (hasClickedSend) {
                    clearInterval(clickInterval);
                    return;
                }
            }
        }
    }, 400); // הקטנתי את ההשהייה כדי להיות מהירים יותר ברגע שהטקסט מגיע
}

function showToast(msg) {
    // מניעת כפילויות של Toast
    const existing = document.getElementById('gem-fixer-toast');
    if (existing) existing.remove();

    const div = document.createElement('div');
    div.id = 'gem-fixer-toast';
    div.textContent = msg;
    // עיצוב משופר
    div.style.cssText = `
        position: fixed; 
        top: 24px; 
        left: 50%;
        transform: translateX(-50%);
        background: #1e1e1e; 
        color: #e3e3e3; 
        padding: 12px 20px; 
        border-radius: 8px; 
        z-index: 99999; 
        font-family: 'Google Sans', sans-serif; 
        font-size: 14px; 
        box-shadow: 0 4px 15px rgba(0,0,0,0.3); 
        border: 1px solid #444;
        direction: rtl;
        transition: opacity 0.3s ease;
    `;
    document.body.appendChild(div);
    setTimeout(() => {
        div.style.opacity = '0';
        setTimeout(() => div.remove(), 300);
    }, 3000);
}

// =============================================================================
// Observer עם Debounce (ביצועים)
// =============================================================================

const observer = new MutationObserver(() => {
    // מונע הרצה של הלוגיקה על כל פיקסל שמשתנה
    if (debounceTimer) clearTimeout(debounceTimer);

    debounceTimer = setTimeout(() => {
        if (location.href.includes('/gems/edit/')) {
            scrapeEditPage();
        } else {
            handleChatPage();
        }
    }, 200); // רץ רק אחרי שה-DOM נרגע ל-200ms
});

// התחלה
if (document.body) {
    observer.observe(document.body, { childList: true, subtree: true });
} else {
    window.addEventListener('DOMContentLoaded', () => {
        observer.observe(document.body, { childList: true, subtree: true });
    });
}